/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.commons.auth.authorizer;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import org.apache.iotdb.commons.auth.AuthException;
import org.apache.iotdb.commons.auth.authorizer.IAuthorizer;
import org.apache.iotdb.commons.auth.entity.PrivilegeType;
import org.apache.iotdb.commons.auth.entity.PrivilegeUnion;
import org.apache.iotdb.commons.auth.entity.Role;
import org.apache.iotdb.commons.auth.entity.User;
import org.apache.iotdb.commons.auth.role.BasicRoleManager;
import org.apache.iotdb.commons.auth.user.BasicUserManager;
import org.apache.iotdb.commons.conf.CommonDescriptor;
import org.apache.iotdb.commons.exception.StartupException;
import org.apache.iotdb.commons.path.PartialPath;
import org.apache.iotdb.commons.security.encrypt.AsymmetricEncrypt;
import org.apache.iotdb.commons.service.IService;
import org.apache.iotdb.commons.service.ServiceType;
import org.apache.iotdb.commons.utils.AuthUtils;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class BasicAuthorizer
implements IAuthorizer,
IService {
    private static final Logger LOGGER = LoggerFactory.getLogger(BasicAuthorizer.class);
    private static final String NO_SUCH_ROLE_EXCEPTION = "No such role : %s";
    private static final String NO_SUCH_USER_EXCEPTION = "No such user : %s";
    BasicUserManager userManager;
    BasicRoleManager roleManager;

    public BasicAuthorizer(BasicUserManager userManager, BasicRoleManager roleManager) {
        this.userManager = userManager;
        this.roleManager = roleManager;
    }

    protected void init() throws AuthException {
        this.userManager.reset();
        this.roleManager.reset();
        LOGGER.info("Initialization of Authorizer completes");
    }

    public static IAuthorizer getInstance() throws AuthException {
        if (InstanceHolder.instance == null) {
            throw new AuthException(TSStatusCode.INIT_AUTH_ERROR, "Authorizer uninitialized");
        }
        return InstanceHolder.instance;
    }

    protected abstract boolean isAdmin(String var1);

    private void checkAdmin(String username, String errmsg) throws AuthException {
        if (this.isAdmin(username)) {
            throw new AuthException(TSStatusCode.NO_PERMISSION, errmsg);
        }
    }

    @Override
    public boolean login(String username, String password) throws AuthException {
        User user = this.userManager.getEntity(username);
        if (user == null || password == null) {
            return false;
        }
        if (AuthUtils.validatePassword(password, user.getPassword(), AsymmetricEncrypt.DigestAlgorithm.SHA_256)) {
            return true;
        }
        if (AuthUtils.validatePassword(password, user.getPassword(), AsymmetricEncrypt.DigestAlgorithm.MD5)) {
            this.userManager.updateUserPassword(username, password);
            return true;
        }
        return false;
    }

    @Override
    public String login4Pipe(String username, String password) {
        User user = this.userManager.getEntity(username);
        if (Objects.isNull(password)) {
            return user.getPassword();
        }
        if (user == null) {
            return null;
        }
        if (AuthUtils.validatePassword(password, user.getPassword(), AsymmetricEncrypt.DigestAlgorithm.SHA_256)) {
            return user.getPassword();
        }
        if (AuthUtils.validatePassword(password, user.getPassword(), AsymmetricEncrypt.DigestAlgorithm.MD5)) {
            try {
                this.userManager.updateUserPassword(username, password);
            }
            catch (AuthException authException) {
                // empty catch block
            }
            return this.userManager.getEntity(username).getPassword();
        }
        return null;
    }

    @Override
    public void createUser(String username, String password) throws AuthException {
        if (!this.userManager.createUser(username, password, true, true)) {
            throw new AuthException(TSStatusCode.USER_ALREADY_EXIST, String.format("User %s already exists", username));
        }
    }

    @Override
    public void createUserWithoutCheck(String username, String password) throws AuthException {
        if (!this.userManager.createUser(username, password, false, true)) {
            throw new AuthException(TSStatusCode.USER_ALREADY_EXIST, String.format("User %s already exists", username));
        }
    }

    @Override
    public void createUserWithRawPassword(String username, String password) throws AuthException {
        if (!this.userManager.createUser(username, password, true, false)) {
            throw new AuthException(TSStatusCode.USER_ALREADY_EXIST, String.format("User %s already exists", username));
        }
    }

    @Override
    public void deleteUser(String username) throws AuthException {
        this.checkAdmin(username, "Default administrator cannot be deleted");
        if (!this.userManager.deleteEntity(username)) {
            throw new AuthException(TSStatusCode.USER_NOT_EXIST, String.format("User %s does not exist", username));
        }
    }

    @Override
    public void grantPrivilegeToUser(String username, PrivilegeUnion union) throws AuthException {
        this.checkAdmin(username, "Invalid operation, administrator already has all privileges");
        this.userManager.grantPrivilegeToEntity(username, union);
    }

    @Override
    public void revokePrivilegeFromUser(String username, PrivilegeUnion union) throws AuthException {
        this.checkAdmin(username, "Invalid operation, administrator must have all privileges");
        this.userManager.revokePrivilegeFromEntity(username, union);
    }

    @Override
    public void revokeAllPrivilegeFromUser(String userName) throws AuthException {
        this.checkAdmin(userName, "Invalid operation, administrator cannot revoke privileges");
        User user = this.userManager.getEntity(userName);
        if (user == null) {
            throw new AuthException(TSStatusCode.USER_NOT_EXIST, String.format("User %s does not exist", userName));
        }
        user.revokeAllRelationalPrivileges();
    }

    @Override
    public void createRole(String roleName) throws AuthException {
        AuthUtils.validateRolename(roleName);
        if (!this.roleManager.createRole(roleName)) {
            LOGGER.error("Role {} already exists", (Object)roleName);
            throw new AuthException(TSStatusCode.ROLE_ALREADY_EXIST, String.format("Role %s already exists", roleName));
        }
    }

    @Override
    public void deleteRole(String roleName) throws AuthException {
        boolean success = this.roleManager.deleteEntity(roleName);
        if (!success) {
            throw new AuthException(TSStatusCode.ROLE_NOT_EXIST, String.format("Role %s does not exist", roleName));
        }
        List<String> users = this.userManager.listAllEntities();
        for (String user : users) {
            try {
                this.userManager.revokeRoleFromUser(roleName, user);
            }
            catch (AuthException e) {
                LOGGER.warn("Error encountered when revoking a role {} from user {} after deletion", new Object[]{roleName, user, e});
            }
        }
    }

    @Override
    public void grantPrivilegeToRole(String rolename, PrivilegeUnion union) throws AuthException {
        this.roleManager.grantPrivilegeToEntity(rolename, union);
    }

    @Override
    public void revokePrivilegeFromRole(String roleName, PrivilegeUnion union) throws AuthException {
        this.roleManager.revokePrivilegeFromEntity(roleName, union);
    }

    @Override
    public void revokeAllPrivilegeFromRole(String roleName) throws AuthException {
        Role role = this.roleManager.getEntity(roleName);
        if (role == null) {
            throw new AuthException(TSStatusCode.ROLE_NOT_EXIST, String.format("Role %s does not exist", roleName));
        }
        role.revokeAllRelationalPrivileges();
    }

    @Override
    public void grantRoleToUser(String roleName, String userName) throws AuthException {
        this.checkAdmin(userName, "Invalid operation, cannot grant role to administrator");
        Role role = this.roleManager.getEntity(roleName);
        if (role == null) {
            throw new AuthException(TSStatusCode.ROLE_NOT_EXIST, String.format(NO_SUCH_ROLE_EXCEPTION, roleName));
        }
        this.userManager.grantRoleToUser(roleName, userName);
        role = this.roleManager.getEntity(roleName);
        if (role == null) {
            throw new AuthException(TSStatusCode.ROLE_NOT_EXIST, String.format(NO_SUCH_ROLE_EXCEPTION, roleName));
        }
    }

    @Override
    public void revokeRoleFromUser(String roleName, String userName) throws AuthException {
        if (this.isAdmin(userName)) {
            throw new AuthException(TSStatusCode.NO_PERMISSION, "Invalid operation, cannot revoke role from administrator ");
        }
        Role role = this.roleManager.getEntity(roleName);
        if (role == null) {
            throw new AuthException(TSStatusCode.ROLE_NOT_EXIST, String.format(NO_SUCH_ROLE_EXCEPTION, roleName));
        }
        this.userManager.revokeRoleFromUser(roleName, userName);
    }

    @Override
    public Set<PrivilegeType> getPrivileges(String userName, PartialPath path) throws AuthException {
        User user = this.userManager.getEntity(userName);
        if (user == null) {
            throw new AuthException(TSStatusCode.USER_NOT_EXIST, String.format(NO_SUCH_USER_EXCEPTION, userName));
        }
        Set<PrivilegeType> privileges = user.getPathPrivileges(path);
        for (String roleName : user.getRoleSet()) {
            Role role = this.roleManager.getEntity(roleName);
            if (role == null) continue;
            privileges.addAll(role.getPathPrivileges(path));
        }
        return privileges;
    }

    @Override
    public void updateUserPassword(String userName, String newPassword) throws AuthException {
        if (!this.userManager.updateUserPassword(userName, newPassword)) {
            throw new AuthException(TSStatusCode.ILLEGAL_PARAMETER, "password " + newPassword + " is illegal");
        }
    }

    @Override
    public boolean checkUserPrivileges(String userName, PrivilegeUnion union) throws AuthException {
        if (this.isAdmin(userName)) {
            return true;
        }
        User user = this.userManager.getEntity(userName);
        if (user == null) {
            throw new AuthException(TSStatusCode.USER_NOT_EXIST, String.format(NO_SUCH_USER_EXCEPTION, userName));
        }
        if (this.checkEntityPrivileges(user, union)) {
            return true;
        }
        for (String roleName : user.getRoleSet()) {
            Role role = this.roleManager.getEntity(roleName);
            if (!this.checkEntityPrivileges(role, union)) continue;
            return true;
        }
        return false;
    }

    private boolean checkEntityPrivileges(Role role, PrivilegeUnion union) {
        switch (union.getModelType()) {
            case TREE: {
                if (union.isGrantOption()) {
                    return role.checkPathPrivilegeGrantOpt(union.getPath(), union.getPrivilegeType());
                }
                return role.checkPathPrivilege(union.getPath(), union.getPrivilegeType());
            }
            case RELATIONAL: {
                if (union.isForAny()) {
                    if (union.getPrivilegeType() == null) {
                        return role.checkAnyVisible();
                    }
                    if (union.isGrantOption()) {
                        return role.checkAnyScopePrivilegeGrantOption(union.getPrivilegeType());
                    }
                    return role.checkAnyScopePrivilege(union.getPrivilegeType());
                }
                if (union.getTbName() == null) {
                    if (union.getPrivilegeType() == null) {
                        return role.checkDBVisible(union.getDBName());
                    }
                    if (union.isGrantOption()) {
                        return role.checkDatabasePrivilegeGrantOption(union.getDBName(), union.getPrivilegeType());
                    }
                    return role.checkDatabasePrivilege(union.getDBName(), union.getPrivilegeType());
                }
                if (union.getPrivilegeType() == null) {
                    return role.checkTBVisible(union.getDBName(), union.getTbName());
                }
                if (union.isGrantOption()) {
                    return role.checkTablePrivilegeGrantOption(union.getDBName(), union.getTbName(), union.getPrivilegeType());
                }
                return role.checkTablePrivilege(union.getDBName(), union.getTbName(), union.getPrivilegeType());
            }
            case SYSTEM: {
                if (union.isGrantOption()) {
                    return role.checkSysPriGrantOpt(union.getPrivilegeType());
                }
                return role.checkSysPrivilege(union.getPrivilegeType());
            }
        }
        return false;
    }

    @Override
    public Map<String, User> getAllUsers() {
        HashMap<String, User> allUsers = new HashMap<String, User>();
        List<String> userNames = this.listAllUsers();
        for (String userName : userNames) {
            try {
                allUsers.put(userName, this.getUser(userName));
            }
            catch (AuthException e) {
                LOGGER.error("get all users failed, No such user: {}", (Object)userName);
            }
        }
        return allUsers;
    }

    @Override
    public Map<String, Role> getAllRoles() {
        HashMap<String, Role> allRoles = new HashMap<String, Role>();
        List<String> roleNames = this.listAllRoles();
        for (String roleName : roleNames) {
            try {
                allRoles.put(roleName, this.getRole(roleName));
            }
            catch (AuthException e) {
                LOGGER.error("get all roles failed, No such role: {}", (Object)roleName);
            }
        }
        return allRoles;
    }

    @Override
    public void reset() throws AuthException {
        this.init();
    }

    @Override
    public void start() throws StartupException {
        try {
            this.init();
        }
        catch (AuthException e) {
            throw new StartupException(e);
        }
    }

    @Override
    public void stop() {
    }

    @Override
    public ServiceType getID() {
        return ServiceType.AUTHORIZATION_SERVICE;
    }

    @Override
    public List<String> listAllUsers() {
        return this.userManager.listAllEntities();
    }

    @Override
    public List<String> listAllRoles() {
        return this.roleManager.listAllEntities();
    }

    @Override
    public Role getRole(String roleName) throws AuthException {
        return this.roleManager.getEntity(roleName);
    }

    @Override
    public User getUser(String username) throws AuthException {
        return this.userManager.getEntity(username);
    }

    @Override
    public boolean processTakeSnapshot(File snapshotDir) throws TException, IOException {
        return this.userManager.processTakeSnapshot(snapshotDir) && this.roleManager.processTakeSnapshot(snapshotDir);
    }

    @Override
    public void processLoadSnapshot(File snapshotDir) throws TException, IOException {
        this.userManager.processLoadSnapshot(snapshotDir);
        this.roleManager.processLoadSnapshot(snapshotDir);
    }

    private static class InstanceHolder {
        private static final IAuthorizer instance;

        private InstanceHolder() {
        }

        static {
            try {
                Class<?> c = Class.forName(CommonDescriptor.getInstance().getConfig().getAuthorizerProvider());
                LOGGER.info("Authorizer provider class: {}", (Object)CommonDescriptor.getInstance().getConfig().getAuthorizerProvider());
                instance = (IAuthorizer)c.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            }
            catch (Exception e) {
                throw new IllegalStateException("Authorizer could not be initialized!", e);
            }
        }
    }
}

