/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a;

import com.amazonaws.SdkBaseException;
import com.amazonaws.event.ProgressEvent;
import com.amazonaws.event.ProgressEventType;
import com.amazonaws.event.ProgressListener;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.PutObjectResult;
import com.amazonaws.services.s3.model.UploadPartRequest;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.StringJoiner;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.fs.Abortable;
import org.apache.hadoop.fs.StreamCapabilities;
import org.apache.hadoop.fs.Syncable;
import org.apache.hadoop.fs.s3a.S3ADataBlocks;
import org.apache.hadoop.fs.s3a.S3AUtils;
import org.apache.hadoop.fs.s3a.Statistic;
import org.apache.hadoop.fs.s3a.WriteOperationHelper;
import org.apache.hadoop.fs.s3a.WriteOperations;
import org.apache.hadoop.fs.s3a.commit.PutTracker;
import org.apache.hadoop.fs.s3a.impl.PutObjectOptions;
import org.apache.hadoop.fs.s3a.statistics.BlockOutputStreamStatistics;
import org.apache.hadoop.fs.s3a.statistics.impl.EmptyS3AStatisticsContext;
import org.apache.hadoop.fs.statistics.DurationTracker;
import org.apache.hadoop.fs.statistics.DurationTrackerFactory;
import org.apache.hadoop.fs.statistics.IOStatistics;
import org.apache.hadoop.fs.statistics.IOStatisticsAggregator;
import org.apache.hadoop.fs.statistics.IOStatisticsLogging;
import org.apache.hadoop.fs.statistics.IOStatisticsSource;
import org.apache.hadoop.fs.statistics.impl.IOStatisticsBinding;
import org.apache.hadoop.fs.store.LogExactlyOnce;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.Futures;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListenableFuture;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListeningExecutorService;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.MoreExecutors;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
class S3ABlockOutputStream
extends OutputStream
implements StreamCapabilities,
IOStatisticsSource,
Syncable,
Abortable {
    private static final Logger LOG = LoggerFactory.getLogger(S3ABlockOutputStream.class);
    private static final String E_NOT_SYNCABLE = "S3A streams are not Syncable. See HADOOP-17597.";
    private final String key;
    private final long blockSize;
    private final IOStatistics iostatistics;
    private final BlockOutputStreamBuilder builder;
    private long bytesSubmitted;
    private final ProgressListener progressListener;
    private final ListeningExecutorService executorService;
    private final S3ADataBlocks.BlockFactory blockFactory;
    private final byte[] singleCharWrite = new byte[1];
    private MultiPartUpload multiPartUpload;
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private S3ADataBlocks.DataBlock activeBlock;
    private long blockCount = 0L;
    private final BlockOutputStreamStatistics statistics;
    private final WriteOperations writeOperationHelper;
    private final PutTracker putTracker;
    private final boolean downgradeSyncableExceptions;
    private static final LogExactlyOnce WARN_ON_SYNCABLE = new LogExactlyOnce(LOG);
    private final boolean isCSEEnabled;
    private final IOStatisticsAggregator threadIOStatisticsAggregator;
    private final boolean isMultipartUploadEnabled;

    S3ABlockOutputStream(BlockOutputStreamBuilder builder) throws IOException {
        builder.validate();
        this.builder = builder;
        this.key = builder.key;
        this.blockFactory = builder.blockFactory;
        this.statistics = builder.statistics;
        this.iostatistics = this.statistics.getIOStatistics();
        this.writeOperationHelper = builder.writeOperations;
        this.putTracker = builder.putTracker;
        this.executorService = MoreExecutors.listeningDecorator((ExecutorService)builder.executorService);
        this.multiPartUpload = null;
        Progressable progress = builder.progress;
        this.progressListener = progress instanceof ProgressListener ? (ProgressListener)progress : new ProgressableListener(progress);
        this.downgradeSyncableExceptions = builder.downgradeSyncableExceptions;
        this.isMultipartUploadEnabled = builder.isMultipartUploadEnabled;
        long l = this.blockSize = this.isMultipartUploadEnabled ? builder.blockSize : -1L;
        if (this.putTracker.initialize()) {
            LOG.debug("Put tracker requests multipart upload");
            this.initMultipartUpload();
        }
        this.isCSEEnabled = builder.isCSEEnabled;
        this.threadIOStatisticsAggregator = builder.ioStatisticsAggregator;
        this.createBlockIfNeeded();
        LOG.debug("Initialized S3ABlockOutputStream for {} output to {}", (Object)this.key, (Object)this.activeBlock);
    }

    private synchronized S3ADataBlocks.DataBlock createBlockIfNeeded() throws IOException {
        if (this.activeBlock == null) {
            ++this.blockCount;
            if (this.blockCount >= 10000L) {
                LOG.error("Number of partitions in stream exceeds limit for S3: 10000 write may fail.");
            }
            this.activeBlock = this.blockFactory.create(this.blockCount, this.blockSize, this.statistics);
        }
        return this.activeBlock;
    }

    private synchronized S3ADataBlocks.DataBlock getActiveBlock() {
        return this.activeBlock;
    }

    private synchronized boolean hasActiveBlock() {
        return this.activeBlock != null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void clearActiveBlock() {
        if (this.activeBlock != null) {
            LOG.debug("Clearing active block");
        }
        S3ABlockOutputStream s3ABlockOutputStream = this;
        synchronized (s3ABlockOutputStream) {
            this.activeBlock = null;
        }
    }

    void checkOpen() throws IOException {
        if (this.closed.get()) {
            throw new IOException("Filesystem " + this.writeOperationHelper + " closed");
        }
    }

    @Override
    public synchronized void flush() throws IOException {
        try {
            this.checkOpen();
        }
        catch (IOException e) {
            LOG.warn("Stream closed: " + e.getMessage());
            return;
        }
        S3ADataBlocks.DataBlock dataBlock = this.getActiveBlock();
        if (dataBlock != null) {
            dataBlock.flush();
        }
    }

    @Override
    public synchronized void write(int b) throws IOException {
        this.singleCharWrite[0] = (byte)b;
        this.write(this.singleCharWrite, 0, 1);
    }

    @Override
    public synchronized void write(byte[] source, int offset, int len) throws IOException {
        S3ADataBlocks.validateWriteArgs(source, offset, len);
        this.checkOpen();
        if (len == 0) {
            return;
        }
        this.statistics.writeBytes(len);
        S3ADataBlocks.DataBlock block = this.createBlockIfNeeded();
        int written = block.write(source, offset, len);
        if (!this.isMultipartUploadEnabled) {
            return;
        }
        int remainingCapacity = (int)block.remainingCapacity();
        if (written < len) {
            LOG.debug("writing more data than block has capacity -triggering upload");
            this.uploadCurrentBlock(false);
            this.write(source, offset + written, len - written);
        } else if (remainingCapacity == 0 && !this.isCSEEnabled) {
            this.uploadCurrentBlock(false);
        }
    }

    private synchronized void uploadCurrentBlock(boolean isLast) throws IOException {
        Preconditions.checkState((boolean)this.hasActiveBlock(), (Object)"No active block");
        LOG.debug("Writing block # {}", (Object)this.blockCount);
        this.initMultipartUpload();
        try {
            this.multiPartUpload.uploadBlockAsync(this.getActiveBlock(), isLast);
            this.bytesSubmitted += this.getActiveBlock().dataSize();
        }
        finally {
            this.clearActiveBlock();
        }
    }

    private void initMultipartUpload() throws IOException {
        Preconditions.checkState((boolean)this.isMultipartUploadEnabled, (Object)"multipart upload is disabled");
        if (this.multiPartUpload == null) {
            LOG.debug("Initiating Multipart upload");
            this.multiPartUpload = new MultiPartUpload(this.key);
        }
    }

    @Override
    public void close() throws IOException {
        if (this.closed.getAndSet(true)) {
            LOG.debug("Ignoring close() as stream is already closed");
            return;
        }
        S3ADataBlocks.DataBlock block = this.getActiveBlock();
        boolean hasBlock = this.hasActiveBlock();
        LOG.debug("{}: Closing block #{}: current block= {}", new Object[]{this, this.blockCount, hasBlock ? block : "(none)"});
        long bytes = 0L;
        try {
            if (this.multiPartUpload == null) {
                if (hasBlock) {
                    this.bytesSubmitted = bytes = this.putObject();
                }
            } else {
                if (hasBlock && (block.hasData() || this.multiPartUpload.getPartsSubmitted() == 0)) {
                    this.uploadCurrentBlock(true);
                }
                List partETags = this.multiPartUpload.waitForAllPartUploads();
                bytes = this.bytesSubmitted;
                if (this.putTracker.aboutToComplete(this.multiPartUpload.getUploadId(), partETags, bytes, this.iostatistics)) {
                    this.multiPartUpload.complete(partETags);
                } else {
                    LOG.info("File {} will be visible when the job is committed", (Object)this.key);
                }
            }
            if (!this.putTracker.outputImmediatelyVisible()) {
                this.statistics.commitUploaded(bytes);
            }
            LOG.debug("Upload complete to {} by {}", (Object)this.key, (Object)this.writeOperationHelper);
        }
        catch (IOException ioe) {
            this.maybeAbortMultipart();
            this.writeOperationHelper.writeFailed(ioe);
            throw ioe;
        }
        finally {
            this.cleanupOnClose();
        }
        this.writeOperationHelper.writeSuccessful(bytes);
    }

    private synchronized void cleanupOnClose() {
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{this.getActiveBlock(), this.blockFactory});
        this.mergeThreadIOStatistics(this.statistics.getIOStatistics());
        LOG.debug("Statistics: {}", (Object)this.statistics);
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{this.statistics});
        this.clearActiveBlock();
    }

    private void mergeThreadIOStatistics(IOStatistics streamStatistics) {
        this.getThreadIOStatistics().aggregate(streamStatistics);
    }

    private synchronized IOException maybeAbortMultipart() {
        if (this.multiPartUpload != null) {
            IOException ioe = this.multiPartUpload.abort();
            this.multiPartUpload = null;
            return ioe;
        }
        return null;
    }

    /*
     * Loose catch block
     */
    public Abortable.AbortableResult abort() {
        if (this.closed.getAndSet(true)) {
            LOG.debug("Ignoring abort() as stream is already closed");
            return new AbortableResultImpl(true, null);
        }
        try {
            try (DurationTracker d = this.statistics.trackDuration(Statistic.INVOCATION_ABORT.getSymbol());){
                AbortableResultImpl abortableResultImpl = new AbortableResultImpl(false, this.maybeAbortMultipart());
                return abortableResultImpl;
            }
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            this.cleanupOnClose();
        }
    }

    private long putObject() throws IOException {
        LOG.debug("Executing regular upload for {}", (Object)this.writeOperationHelper);
        S3ADataBlocks.DataBlock block = this.getActiveBlock();
        long size = block.dataSize();
        S3ADataBlocks.BlockUploadData uploadData = block.startUpload();
        PutObjectRequest putObjectRequest = uploadData.hasFile() ? this.writeOperationHelper.createPutObjectRequest(this.key, uploadData.getFile(), this.builder.putOptions) : this.writeOperationHelper.createPutObjectRequest(this.key, uploadData.getUploadStream(), size, this.builder.putOptions);
        BlockUploadProgress callback = new BlockUploadProgress(block, this.progressListener, this.now());
        putObjectRequest.setGeneralProgressListener((ProgressListener)callback);
        this.statistics.blockUploadQueued(size);
        ListenableFuture putObjectResult = this.executorService.submit(() -> {
            PutObjectResult putObjectResult;
            try {
                putObjectResult = this.writeOperationHelper.putObject(putObjectRequest, this.builder.putOptions, this.statistics);
            }
            catch (Throwable throwable) {
                IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{uploadData, block});
                throw throwable;
            }
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{uploadData, block});
            return putObjectResult;
        });
        this.clearActiveBlock();
        try {
            putObjectResult.get();
            return size;
        }
        catch (InterruptedException ie) {
            LOG.warn("Interrupted object upload", (Throwable)ie);
            Thread.currentThread().interrupt();
            return 0L;
        }
        catch (ExecutionException ee) {
            throw S3AUtils.extractException("regular upload", this.key, ee);
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("S3ABlockOutputStream{");
        sb.append(this.writeOperationHelper.toString());
        sb.append(", blockSize=").append(this.blockSize);
        sb.append(", isMultipartUploadEnabled=").append(this.isMultipartUploadEnabled);
        S3ADataBlocks.DataBlock block = this.activeBlock;
        if (block != null) {
            sb.append(", activeBlock=").append(block);
        }
        sb.append(" Statistics=").append(IOStatisticsLogging.ioStatisticsSourceToString((Object)this));
        sb.append('}');
        return sb.toString();
    }

    private void incrementWriteOperations() {
        this.writeOperationHelper.incrementWriteOperations();
    }

    private Instant now() {
        return Instant.now();
    }

    BlockOutputStreamStatistics getStatistics() {
        return this.statistics;
    }

    public boolean hasCapability(String capability) {
        switch (capability.toLowerCase(Locale.ENGLISH)) {
            case "fs.s3a.capability.magic.output.stream": 
            case "s3a:magic.output.stream": {
                return !this.putTracker.outputImmediatelyVisible();
            }
            case "hflush": 
            case "hsync": {
                return false;
            }
            case "iostatistics": {
                return true;
            }
            case "fs.capability.outputstream.abortable": {
                return true;
            }
            case "fs.capability.iocontext.supported": {
                return true;
            }
        }
        return false;
    }

    public void hflush() throws IOException {
        this.statistics.hflushInvoked();
        this.handleSyncableInvocation();
    }

    public void hsync() throws IOException {
        this.statistics.hsyncInvoked();
        this.handleSyncableInvocation();
    }

    private void handleSyncableInvocation() {
        UnsupportedOperationException ex = new UnsupportedOperationException(E_NOT_SYNCABLE);
        if (!this.downgradeSyncableExceptions) {
            throw ex;
        }
        WARN_ON_SYNCABLE.warn("Application invoked the Syncable API against stream writing to {}. This is Unsupported", new Object[]{this.key});
        LOG.debug("Downgrading Syncable call", (Throwable)ex);
    }

    public IOStatistics getIOStatistics() {
        return this.iostatistics;
    }

    protected IOStatisticsAggregator getThreadIOStatistics() {
        return this.threadIOStatisticsAggregator;
    }

    public static BlockOutputStreamBuilder builder() {
        return new BlockOutputStreamBuilder();
    }

    public static final class BlockOutputStreamBuilder {
        private String key;
        private ExecutorService executorService;
        private Progressable progress;
        private long blockSize;
        private S3ADataBlocks.BlockFactory blockFactory;
        private BlockOutputStreamStatistics statistics = EmptyS3AStatisticsContext.EMPTY_BLOCK_OUTPUT_STREAM_STATISTICS;
        private WriteOperations writeOperations;
        private PutTracker putTracker;
        private boolean downgradeSyncableExceptions;
        private boolean isCSEEnabled;
        private PutObjectOptions putOptions;
        private IOStatisticsAggregator ioStatisticsAggregator;
        private boolean isMultipartUploadEnabled;

        private BlockOutputStreamBuilder() {
        }

        public void validate() {
            Objects.requireNonNull(this.key, "null key");
            Objects.requireNonNull(this.executorService, "null executorService");
            Objects.requireNonNull(this.blockFactory, "null blockFactory");
            Objects.requireNonNull(this.statistics, "null statistics");
            Objects.requireNonNull(this.writeOperations, "null writeOperationHelper");
            Objects.requireNonNull(this.putTracker, "null putTracker");
            Objects.requireNonNull(this.putOptions, "null putOptions");
            Preconditions.checkArgument((this.blockSize >= 0x500000L ? 1 : 0) != 0, (String)"Block size is too small: %s", (Object[])new Object[]{this.blockSize});
            Objects.requireNonNull(this.ioStatisticsAggregator, "null ioStatisticsAggregator");
        }

        public BlockOutputStreamBuilder withKey(String value) {
            this.key = value;
            return this;
        }

        public BlockOutputStreamBuilder withExecutorService(ExecutorService value) {
            this.executorService = value;
            return this;
        }

        public BlockOutputStreamBuilder withProgress(Progressable value) {
            this.progress = value;
            return this;
        }

        public BlockOutputStreamBuilder withBlockSize(long value) {
            this.blockSize = value;
            return this;
        }

        public BlockOutputStreamBuilder withBlockFactory(S3ADataBlocks.BlockFactory value) {
            this.blockFactory = value;
            return this;
        }

        public BlockOutputStreamBuilder withStatistics(BlockOutputStreamStatistics value) {
            this.statistics = value;
            return this;
        }

        public BlockOutputStreamBuilder withWriteOperations(WriteOperationHelper value) {
            this.writeOperations = value;
            return this;
        }

        public BlockOutputStreamBuilder withPutTracker(PutTracker value) {
            this.putTracker = value;
            return this;
        }

        public BlockOutputStreamBuilder withDowngradeSyncableExceptions(boolean value) {
            this.downgradeSyncableExceptions = value;
            return this;
        }

        public BlockOutputStreamBuilder withCSEEnabled(boolean value) {
            this.isCSEEnabled = value;
            return this;
        }

        public BlockOutputStreamBuilder withPutOptions(PutObjectOptions value) {
            this.putOptions = value;
            return this;
        }

        public BlockOutputStreamBuilder withIOStatisticsAggregator(IOStatisticsAggregator value) {
            this.ioStatisticsAggregator = value;
            return this;
        }

        public BlockOutputStreamBuilder withMultipartEnabled(boolean value) {
            this.isMultipartUploadEnabled = value;
            return this;
        }
    }

    private static class ProgressableListener
    implements ProgressListener {
        private final Progressable progress;

        ProgressableListener(Progressable progress) {
            this.progress = progress;
        }

        public void progressChanged(ProgressEvent progressEvent) {
            if (this.progress != null) {
                this.progress.progress();
            }
        }
    }

    private final class BlockUploadProgress
    implements ProgressListener {
        private final S3ADataBlocks.DataBlock block;
        private final ProgressListener nextListener;
        private final Instant transferQueueTime;
        private Instant transferStartTime;

        private BlockUploadProgress(S3ADataBlocks.DataBlock block, ProgressListener nextListener, Instant transferQueueTime) {
            this.block = block;
            this.transferQueueTime = transferQueueTime;
            this.nextListener = nextListener;
        }

        public void progressChanged(ProgressEvent progressEvent) {
            ProgressEventType eventType = progressEvent.getEventType();
            long bytesTransferred = progressEvent.getBytesTransferred();
            long size = this.block.dataSize();
            switch (eventType) {
                case REQUEST_BYTE_TRANSFER_EVENT: {
                    S3ABlockOutputStream.this.statistics.bytesTransferred(bytesTransferred);
                    break;
                }
                case TRANSFER_PART_STARTED_EVENT: {
                    this.transferStartTime = S3ABlockOutputStream.this.now();
                    S3ABlockOutputStream.this.statistics.blockUploadStarted(Duration.between(this.transferQueueTime, this.transferStartTime), size);
                    S3ABlockOutputStream.this.incrementWriteOperations();
                    break;
                }
                case TRANSFER_PART_COMPLETED_EVENT: {
                    S3ABlockOutputStream.this.statistics.blockUploadCompleted(Duration.between(this.transferStartTime, S3ABlockOutputStream.this.now()), size);
                    break;
                }
                case TRANSFER_PART_FAILED_EVENT: {
                    S3ABlockOutputStream.this.statistics.blockUploadFailed(Duration.between(this.transferStartTime, S3ABlockOutputStream.this.now()), size);
                    LOG.warn("Transfer failure of block {}", (Object)this.block);
                    break;
                }
            }
            if (this.nextListener != null) {
                this.nextListener.progressChanged(progressEvent);
            }
        }
    }

    private class MultiPartUpload {
        private final String uploadId;
        private final List<ListenableFuture<PartETag>> partETagsFutures;
        private int partsSubmitted;
        private int partsUploaded;
        private long bytesSubmitted;
        private IOException blockUploadFailure;

        MultiPartUpload(String key) throws IOException {
            this.uploadId = (String)IOStatisticsBinding.trackDuration((DurationTrackerFactory)S3ABlockOutputStream.this.statistics, (String)Statistic.OBJECT_MULTIPART_UPLOAD_INITIATED.getSymbol(), () -> S3ABlockOutputStream.this.writeOperationHelper.initiateMultiPartUpload(key, S3ABlockOutputStream.this.builder.putOptions));
            this.partETagsFutures = new ArrayList<ListenableFuture<PartETag>>(2);
            LOG.debug("Initiated multi-part upload for {} with id '{}'", (Object)S3ABlockOutputStream.this.writeOperationHelper, (Object)this.uploadId);
        }

        public int getPartsSubmitted() {
            return this.partsSubmitted;
        }

        public int getPartsUploaded() {
            return this.partsUploaded;
        }

        public String getUploadId() {
            return this.uploadId;
        }

        public long getBytesSubmitted() {
            return this.bytesSubmitted;
        }

        public void noteUploadFailure(IOException e) {
            if (this.blockUploadFailure == null) {
                this.blockUploadFailure = e;
            }
        }

        public void maybeRethrowUploadFailure() throws IOException {
            if (this.blockUploadFailure != null) {
                throw this.blockUploadFailure;
            }
        }

        private void uploadBlockAsync(S3ADataBlocks.DataBlock block, Boolean isLast) throws IOException {
            UploadPartRequest request;
            S3ADataBlocks.BlockUploadData uploadData;
            LOG.debug("Queueing upload of {} for upload {}", (Object)block, (Object)this.uploadId);
            Preconditions.checkNotNull((Object)this.uploadId, (Object)"Null uploadId");
            this.maybeRethrowUploadFailure();
            ++this.partsSubmitted;
            long size = block.dataSize();
            this.bytesSubmitted += size;
            int currentPartNumber = this.partETagsFutures.size() + 1;
            try {
                uploadData = block.startUpload();
                request = S3ABlockOutputStream.this.writeOperationHelper.newUploadPartRequest(S3ABlockOutputStream.this.key, this.uploadId, currentPartNumber, size, uploadData.getUploadStream(), uploadData.getFile(), 0L);
                request.setLastPart(isLast.booleanValue());
            }
            catch (SdkBaseException aws) {
                IOException e = S3AUtils.translateException("upload", S3ABlockOutputStream.this.key, aws);
                this.noteUploadFailure(e);
                throw e;
            }
            catch (IOException e) {
                this.noteUploadFailure(e);
                throw e;
            }
            BlockUploadProgress callback = new BlockUploadProgress(block, S3ABlockOutputStream.this.progressListener, S3ABlockOutputStream.this.now());
            request.setGeneralProgressListener((ProgressListener)callback);
            S3ABlockOutputStream.this.statistics.blockUploadQueued(block.dataSize());
            ListenableFuture partETagFuture = S3ABlockOutputStream.this.executorService.submit(() -> {
                PartETag partETag;
                try {
                    LOG.debug("Uploading part {} for id '{}'", (Object)currentPartNumber, (Object)this.uploadId);
                    PartETag partETag2 = S3ABlockOutputStream.this.writeOperationHelper.uploadPart(request, S3ABlockOutputStream.this.statistics).getPartETag();
                    LOG.debug("Completed upload of {} to part {}", (Object)block, (Object)partETag2.getETag());
                    LOG.debug("Stream statistics of {}", (Object)S3ABlockOutputStream.this.statistics);
                    ++this.partsUploaded;
                    partETag = partETag2;
                }
                catch (IOException e) {
                    try {
                        this.noteUploadFailure(e);
                        throw e;
                    }
                    catch (Throwable throwable) {
                        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{uploadData, block});
                        throw throwable;
                    }
                }
                IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{uploadData, block});
                return partETag;
            });
            this.partETagsFutures.add((ListenableFuture<PartETag>)partETagFuture);
        }

        private List<PartETag> waitForAllPartUploads() throws IOException {
            LOG.debug("Waiting for {} uploads to complete", (Object)this.partETagsFutures.size());
            try {
                return (List)Futures.allAsList(this.partETagsFutures).get();
            }
            catch (InterruptedException ie) {
                LOG.warn("Interrupted partUpload", (Throwable)ie);
                Thread.currentThread().interrupt();
                return null;
            }
            catch (ExecutionException ee) {
                LOG.debug("While waiting for upload completion", (Throwable)ee);
                this.abort();
                throw S3AUtils.extractException("Multi-part upload with id '" + this.uploadId + "' to " + S3ABlockOutputStream.this.key, S3ABlockOutputStream.this.key, ee);
            }
        }

        private void cancelAllActiveFutures() {
            LOG.debug("Cancelling futures");
            for (ListenableFuture<PartETag> future : this.partETagsFutures) {
                future.cancel(true);
            }
        }

        private void complete(List<PartETag> partETags) throws IOException {
            this.maybeRethrowUploadFailure();
            AtomicInteger errorCount = new AtomicInteger(0);
            try {
                IOStatisticsBinding.trackDurationOfInvocation((DurationTrackerFactory)S3ABlockOutputStream.this.statistics, (String)Statistic.MULTIPART_UPLOAD_COMPLETED.getSymbol(), () -> S3ABlockOutputStream.this.writeOperationHelper.completeMPUwithRetries(S3ABlockOutputStream.this.key, this.uploadId, partETags, this.bytesSubmitted, errorCount, S3ABlockOutputStream.this.builder.putOptions));
            }
            finally {
                S3ABlockOutputStream.this.statistics.exceptionInMultipartComplete(errorCount.get());
            }
        }

        private IOException abort() {
            LOG.debug("Aborting upload");
            try {
                IOStatisticsBinding.trackDurationOfInvocation((DurationTrackerFactory)S3ABlockOutputStream.this.statistics, (String)Statistic.OBJECT_MULTIPART_UPLOAD_ABORTED.getSymbol(), () -> {
                    this.cancelAllActiveFutures();
                    S3ABlockOutputStream.this.writeOperationHelper.abortMultipartUpload(S3ABlockOutputStream.this.key, this.uploadId, false, null);
                });
                return null;
            }
            catch (IOException e) {
                LOG.warn("Unable to abort multipart upload, you may need to purge uploaded parts", (Throwable)e);
                S3ABlockOutputStream.this.statistics.exceptionInMultipartAbort();
                return e;
            }
        }
    }

    private static final class AbortableResultImpl
    implements Abortable.AbortableResult {
        private final boolean alreadyClosed;
        private final IOException anyCleanupException;

        private AbortableResultImpl(boolean alreadyClosed, IOException anyCleanupException) {
            this.alreadyClosed = alreadyClosed;
            this.anyCleanupException = anyCleanupException;
        }

        public boolean alreadyClosed() {
            return this.alreadyClosed;
        }

        public IOException anyCleanupException() {
            return this.anyCleanupException;
        }

        public String toString() {
            return new StringJoiner(", ", AbortableResultImpl.class.getSimpleName() + "[", "]").add("alreadyClosed=" + this.alreadyClosed).add("anyCleanupException=" + this.anyCleanupException).toString();
        }
    }
}

