/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.cluster.log;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.iotdb.cluster.config.ClusterDescriptor;
import org.apache.iotdb.cluster.log.Log;
import org.apache.iotdb.cluster.rpc.thrift.AppendEntriesRequest;
import org.apache.iotdb.cluster.rpc.thrift.AppendEntryRequest;
import org.apache.iotdb.cluster.rpc.thrift.Node;
import org.apache.iotdb.cluster.rpc.thrift.RaftService;
import org.apache.iotdb.cluster.server.handlers.caller.AppendNodeEntryHandler;
import org.apache.iotdb.cluster.server.member.RaftMember;
import org.apache.iotdb.cluster.server.monitor.Peer;
import org.apache.iotdb.cluster.server.monitor.Timer;
import org.apache.iotdb.cluster.utils.ClientUtils;
import org.apache.iotdb.db.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.thrift.TException;
import org.apache.thrift.async.AsyncMethodCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LogDispatcher {
    private static final Logger logger = LoggerFactory.getLogger(LogDispatcher.class);
    private RaftMember member;
    private boolean useBatchInLogCatchUp = ClusterDescriptor.getInstance().getConfig().isUseBatchInLogCatchUp();
    private List<BlockingQueue<SendLogRequest>> nodeLogQueues = new ArrayList<BlockingQueue<SendLogRequest>>();
    private ExecutorService executorService;
    private static final ExecutorService serializationService = IoTDBThreadPoolFactory.newFixedThreadPoolWithDaemonThread((int)Runtime.getRuntime().availableProcessors(), (String)"DispatcherEncoder");

    public LogDispatcher(RaftMember member) {
        this.member = member;
        this.executorService = IoTDBThreadPoolFactory.newCachedThreadPool((String)("LogDispatcher-" + member.getName()));
        for (Node node : member.getAllNodes()) {
            if (node.equals(member.getThisNode())) continue;
            this.nodeLogQueues.add(this.createQueueAndBindingThread(node));
        }
    }

    public void close() throws InterruptedException {
        this.executorService.shutdownNow();
        this.executorService.awaitTermination(10L, TimeUnit.SECONDS);
    }

    public void offer(SendLogRequest log) {
        if (!this.nodeLogQueues.isEmpty()) {
            log.serializedLogFuture = LogDispatcher.serializationService.submit(() -> {
                ByteBuffer byteBuffer = log.getLog().serialize();
                log.getLog().setByteSize(byteBuffer.array().length);
                return byteBuffer;
            });
        }
        for (int i = 0; i < this.nodeLogQueues.size(); ++i) {
            BlockingQueue<SendLogRequest> nodeLogQueue = this.nodeLogQueues.get(i);
            try {
                boolean addSucceeded = ClusterDescriptor.getInstance().getConfig().isWaitForSlowNode() ? nodeLogQueue.offer(log, ClusterDescriptor.getInstance().getConfig().getWriteOperationTimeoutMS(), TimeUnit.MILLISECONDS) : nodeLogQueue.add(log);
                if (!addSucceeded) {
                    logger.debug("Log queue[{}] of {} is full, ignore the log to this node", (Object)i, (Object)this.member.getName());
                    continue;
                }
                log.setEnqueueTime(System.nanoTime());
                continue;
            }
            catch (IllegalStateException e) {
                logger.debug("Log queue[{}] of {} is full, ignore the log to this node", (Object)i, (Object)this.member.getName());
                continue;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    private BlockingQueue<SendLogRequest> createQueueAndBindingThread(Node node) {
        ArrayBlockingQueue<SendLogRequest> logBlockingQueue = new ArrayBlockingQueue<SendLogRequest>(ClusterDescriptor.getInstance().getConfig().getMaxNumOfLogsInMem());
        int bindingThreadNum = 1;
        for (int i = 0; i < bindingThreadNum; ++i) {
            this.executorService.submit(new DispatcherThread(node, logBlockingQueue));
        }
        return logBlockingQueue;
    }

    class DispatcherThread
    implements Runnable {
        private Node receiver;
        private BlockingQueue<SendLogRequest> logBlockingDeque;
        private List<SendLogRequest> currBatch = new ArrayList<SendLogRequest>();
        private Peer peer;

        DispatcherThread(Node receiver, BlockingQueue<SendLogRequest> logBlockingDeque) {
            this.receiver = receiver;
            this.logBlockingDeque = logBlockingDeque;
            this.peer = LogDispatcher.this.member.getPeerMap().computeIfAbsent(receiver, r -> new Peer(LogDispatcher.this.member.getLogManager().getLastLogIndex()));
        }

        @Override
        public void run() {
            Thread.currentThread().setName("LogDispatcher-" + LogDispatcher.this.member.getName() + "-" + this.receiver);
            try {
                while (!Thread.interrupted()) {
                    SendLogRequest poll = this.logBlockingDeque.take();
                    this.currBatch.add(poll);
                    this.logBlockingDeque.drainTo(this.currBatch);
                    if (logger.isDebugEnabled()) {
                        logger.debug("Sending {} logs to {}", (Object)this.currBatch.size(), (Object)this.receiver);
                    }
                    for (SendLogRequest request : this.currBatch) {
                        request.getAppendEntryRequest().entry = (ByteBuffer)request.serializedLogFuture.get();
                    }
                    this.sendBatchLogs(this.currBatch);
                    this.currBatch.clear();
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (Exception e) {
                logger.error("Unexpected error in log dispatcher", (Throwable)e);
            }
            logger.info("Dispatcher exits");
        }

        private void appendEntriesAsync(List<ByteBuffer> logList, AppendEntriesRequest request, List<SendLogRequest> currBatch) throws TException {
            AppendEntriesHandler handler = new AppendEntriesHandler(currBatch);
            RaftService.AsyncClient client = LogDispatcher.this.member.getSendLogAsyncClient(this.receiver);
            if (logger.isDebugEnabled()) {
                logger.debug("{}: append entries {} with {} logs", new Object[]{LogDispatcher.this.member.getName(), this.receiver, logList.size()});
            }
            if (client != null) {
                client.appendEntries(request, (AsyncMethodCallback)handler);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void appendEntriesSync(List<ByteBuffer> logList, AppendEntriesRequest request, List<SendLogRequest> currBatch) {
            long startTime = Timer.Statistic.RAFT_SENDER_WAIT_FOR_PREV_LOG.getOperationStartTime();
            if (!LogDispatcher.this.member.waitForPrevLog(this.peer, currBatch.get(0).getLog())) {
                logger.warn("{}: node {} timed out when appending {}", new Object[]{LogDispatcher.this.member.getName(), this.receiver, currBatch.get(0).getLog()});
                return;
            }
            Timer.Statistic.RAFT_SENDER_WAIT_FOR_PREV_LOG.calOperationCostTimeFromStart(startTime);
            RaftService.Client client = LogDispatcher.this.member.getSyncClient(this.receiver);
            if (client == null) {
                logger.error("No available client for {}", (Object)this.receiver);
                return;
            }
            AppendEntriesHandler handler = new AppendEntriesHandler(currBatch);
            startTime = Timer.Statistic.RAFT_SENDER_SEND_LOG.getOperationStartTime();
            try {
                long result = client.appendEntries(request);
                Timer.Statistic.RAFT_SENDER_SEND_LOG.calOperationCostTimeFromStart(startTime);
                if (result != -1L && logger.isInfoEnabled()) {
                    logger.info("{}: Append {} logs to {}, resp: {}", new Object[]{LogDispatcher.this.member.getName(), logList.size(), this.receiver, result});
                }
                handler.onComplete(result);
            }
            catch (TException e) {
                client.getInputProtocol().getTransport().close();
                handler.onError((Exception)((Object)e));
                logger.warn("Failed logs: {}, first index: {}", logList, (Object)(request.prevLogIndex + 1L));
            }
            finally {
                ClientUtils.putBackSyncClient(client);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private AppendEntriesRequest prepareRequest(List<ByteBuffer> logList, List<SendLogRequest> currBatch, int firstIndex) {
            AppendEntriesRequest request = new AppendEntriesRequest();
            if (LogDispatcher.this.member.getHeader() != null) {
                request.setHeader(LogDispatcher.this.member.getHeader());
            }
            request.setLeader(LogDispatcher.this.member.getThisNode());
            request.setLeaderCommit(LogDispatcher.this.member.getLogManager().getCommitLogIndex());
            AtomicLong atomicLong = LogDispatcher.this.member.getTerm();
            synchronized (atomicLong) {
                request.setTerm(LogDispatcher.this.member.getTerm().get());
            }
            request.setEntries(logList);
            request.setPrevLogIndex(currBatch.get(firstIndex).getLog().getCurrLogIndex() - 1L);
            try {
                request.setPrevLogTerm(currBatch.get((int)firstIndex).getAppendEntryRequest().prevLogTerm);
            }
            catch (Exception e) {
                logger.error("getTerm failed for newly append entries", (Throwable)e);
            }
            return request;
        }

        private void sendLogs(List<SendLogRequest> currBatch) throws TException {
            int logIndex = 0;
            logger.debug("send logs from index {} to {}", (Object)currBatch.get(0).getLog().getCurrLogIndex(), (Object)currBatch.get(currBatch.size() - 1).getLog().getCurrLogIndex());
            while (logIndex < currBatch.size()) {
                long curSize;
                long logSize = IoTDBDescriptor.getInstance().getConfig().getThriftMaxFrameSize();
                ArrayList<ByteBuffer> logList = new ArrayList<ByteBuffer>();
                int prevIndex = logIndex;
                while (logIndex < currBatch.size() && logSize - (curSize = (long)currBatch.get((int)logIndex).getAppendEntryRequest().entry.array().length) > 0x400000L) {
                    logSize -= curSize;
                    Timer.Statistic.LOG_DISPATCHER_LOG_IN_QUEUE.calOperationCostTimeFromStart(currBatch.get(logIndex).getLog().getCreateTime());
                    logList.add(currBatch.get((int)logIndex).getAppendEntryRequest().entry);
                    ++logIndex;
                }
                AppendEntriesRequest appendEntriesRequest = this.prepareRequest(logList, currBatch, prevIndex);
                if (ClusterDescriptor.getInstance().getConfig().isUseAsyncServer()) {
                    this.appendEntriesAsync(logList, appendEntriesRequest, currBatch.subList(prevIndex, logIndex));
                } else {
                    this.appendEntriesSync(logList, appendEntriesRequest, currBatch.subList(prevIndex, logIndex));
                }
                while (prevIndex < logIndex) {
                    Timer.Statistic.LOG_DISPATCHER_FROM_CREATE_TO_END.calOperationCostTimeFromStart(currBatch.get(prevIndex).getLog().getCreateTime());
                    ++prevIndex;
                }
            }
        }

        private void sendBatchLogs(List<SendLogRequest> currBatch) throws TException {
            if (currBatch.size() > 1) {
                if (LogDispatcher.this.useBatchInLogCatchUp) {
                    this.sendLogs(currBatch);
                } else {
                    for (SendLogRequest batch : currBatch) {
                        this.sendLog(batch);
                    }
                }
            } else {
                this.sendLog(currBatch.get(0));
            }
        }

        private void sendLog(SendLogRequest logRequest) {
            Timer.Statistic.LOG_DISPATCHER_LOG_IN_QUEUE.calOperationCostTimeFromStart(logRequest.getLog().getCreateTime());
            LogDispatcher.this.member.sendLogToFollower(logRequest.getLog(), logRequest.getVoteCounter(), this.receiver, logRequest.getLeaderShipStale(), logRequest.getNewLeaderTerm(), logRequest.getAppendEntryRequest());
            Timer.Statistic.LOG_DISPATCHER_FROM_CREATE_TO_END.calOperationCostTimeFromStart(logRequest.getLog().getCreateTime());
        }

        class AppendEntriesHandler
        implements AsyncMethodCallback<Long> {
            private final List<AsyncMethodCallback<Long>> singleEntryHandlers;

            private AppendEntriesHandler(List<SendLogRequest> batch) {
                this.singleEntryHandlers = new ArrayList<AsyncMethodCallback<Long>>(batch.size());
                for (SendLogRequest sendLogRequest : batch) {
                    AppendNodeEntryHandler handler = this.getAppendNodeEntryHandler(sendLogRequest.getLog(), sendLogRequest.getVoteCounter(), DispatcherThread.this.receiver, sendLogRequest.getLeaderShipStale(), sendLogRequest.getNewLeaderTerm(), DispatcherThread.this.peer);
                    this.singleEntryHandlers.add(handler);
                }
            }

            public void onComplete(Long aLong) {
                for (AsyncMethodCallback<Long> singleEntryHandler : this.singleEntryHandlers) {
                    singleEntryHandler.onComplete((Object)aLong);
                }
            }

            public void onError(Exception e) {
                for (AsyncMethodCallback<Long> singleEntryHandler : this.singleEntryHandlers) {
                    singleEntryHandler.onError(e);
                }
            }

            private AppendNodeEntryHandler getAppendNodeEntryHandler(Log log, AtomicInteger voteCounter, Node node, AtomicBoolean leaderShipStale, AtomicLong newLeaderTerm, Peer peer) {
                AppendNodeEntryHandler handler = new AppendNodeEntryHandler();
                handler.setReceiver(node);
                handler.setVoteCounter(voteCounter);
                handler.setLeaderShipStale(leaderShipStale);
                handler.setLog(log);
                handler.setMember(LogDispatcher.this.member);
                handler.setPeer(peer);
                handler.setReceiverTerm(newLeaderTerm);
                return handler;
            }
        }
    }

    public static class SendLogRequest {
        private Log log;
        private AtomicInteger voteCounter;
        private AtomicBoolean leaderShipStale;
        private AtomicLong newLeaderTerm;
        private AppendEntryRequest appendEntryRequest;
        private long enqueueTime;
        private Future<ByteBuffer> serializedLogFuture;

        public SendLogRequest(Log log, AtomicInteger voteCounter, AtomicBoolean leaderShipStale, AtomicLong newLeaderTerm, AppendEntryRequest appendEntryRequest) {
            this.setLog(log);
            this.setVoteCounter(voteCounter);
            this.setLeaderShipStale(leaderShipStale);
            this.setNewLeaderTerm(newLeaderTerm);
            this.setAppendEntryRequest(appendEntryRequest);
        }

        public AtomicInteger getVoteCounter() {
            return this.voteCounter;
        }

        public void setVoteCounter(AtomicInteger voteCounter) {
            this.voteCounter = voteCounter;
        }

        public Log getLog() {
            return this.log;
        }

        public void setLog(Log log) {
            this.log = log;
        }

        public long getEnqueueTime() {
            return this.enqueueTime;
        }

        public void setEnqueueTime(long enqueueTime) {
            this.enqueueTime = enqueueTime;
        }

        public AtomicBoolean getLeaderShipStale() {
            return this.leaderShipStale;
        }

        public void setLeaderShipStale(AtomicBoolean leaderShipStale) {
            this.leaderShipStale = leaderShipStale;
        }

        public AtomicLong getNewLeaderTerm() {
            return this.newLeaderTerm;
        }

        void setNewLeaderTerm(AtomicLong newLeaderTerm) {
            this.newLeaderTerm = newLeaderTerm;
        }

        public AppendEntryRequest getAppendEntryRequest() {
            return this.appendEntryRequest;
        }

        public void setAppendEntryRequest(AppendEntryRequest appendEntryRequest) {
            this.appendEntryRequest = appendEntryRequest;
        }

        public String toString() {
            return "SendLogRequest{log=" + this.log + '}';
        }
    }
}

