/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.webbeans.web.failover;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.webbeans.config.WebBeansContext;
import org.apache.webbeans.spi.FailOverService;

/**
 * {@link Filter} which handles the bean fail over.
 * This extra {@link Filter} is required because
 * {@link javax.servlet.ServletRequestListener#requestDestroyed(javax.servlet.ServletRequestEvent)}
 * will be called after Tomcats session replication and is not in sync with the specs.
 */
public class FailOverFilter implements Filter
{
    protected FailOverService failoverService;

    /**
     * Default constructor.
     */
    public FailOverFilter()
    {
        WebBeansContext webBeansContext = WebBeansContext.currentInstance();
        failoverService = webBeansContext.getService(FailOverService.class);
    }

    @Override
    public void destroy()
    {
        //do nothing
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response,
                         FilterChain chain) throws IOException, ServletException
    {
        try
        {
            chain.doFilter(request, response);
        }
        finally
        {
            if (failoverService != null
                && failoverService.isSupportFailOver()
                && request instanceof HttpServletRequest)
            {
                HttpServletRequest httpRequest = (HttpServletRequest) request;
                HttpSession session = httpRequest.getSession(false);

                if (session != null)
                {
                    failoverService.sessionIsIdle(session);
                }
            }
        }
    }

    @Override
    public void init(FilterConfig config) throws ServletException
    {
        //do nothing
    }
}
