// Copyright 2007, 2010, 2011 The Apache Software Foundation
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package org.apache.tapestry5.ioc.internal;

import org.apache.tapestry5.commons.AnnotationProvider;
import org.apache.tapestry5.commons.ObjectLocator;
import org.apache.tapestry5.ioc.Registry;

import java.lang.annotation.Annotation;

/**
 * A wrapper around {@link InternalRegistry} that exists to expand symbols in a service id before
 * invoking {@link ObjectLocator#getService(String, Class)}.
 */
public class RegistryWrapper implements Registry
{
    private final InternalRegistry registry;

    public RegistryWrapper(final InternalRegistry registry)
    {
        this.registry = registry;
    }

    @Override
    public void cleanupThread()
    {
        registry.cleanupThread();
    }

    @Override
    public void shutdown()
    {
        registry.shutdown();
    }

    @Override
    public <T> T getObject(Class<T> objectType, AnnotationProvider annotationProvider)
    {
        return registry.getObject(objectType, annotationProvider);
    }

    @Override
    public <T> T getService(String serviceId, Class<T> serviceInterface)
    {
        String expandedServiceId = registry.expandSymbols(serviceId);

        return registry.getService(expandedServiceId, serviceInterface);
    }

    @Override
    public <T> T getService(Class<T> serviceInterface)
    {
        return registry.getService(serviceInterface);
    }

    @Override
    public <T> T getService(Class<T> serviceInterface, Class<? extends Annotation>... markerTypes)
    {
        return registry.getService(serviceInterface, markerTypes);
    }

    @Override
    public <T> T autobuild(Class<T> clazz)
    {
        return registry.autobuild(clazz);
    }

    @Override
    public <T> T autobuild(String description, Class<T> clazz)
    {
        return registry.autobuild(description, clazz);
    }

    @Override
    public void performRegistryStartup()
    {
        registry.performRegistryStartup();
    }

    @Override
    public <T> T proxy(Class<T> interfaceClass, Class<? extends T> implementationClass)
    {
        return registry.proxy(interfaceClass, implementationClass);
    }

}
